﻿"""This is a generic transport which is meant to be controlled by a combination
joystick and orientation tracker (or 6DoF tracker). The transports directional
movements, e.g. moveForward, moveBackward, etc. Movement commands are all
relative to the orientation from the tracker. To be clear, this is not
particular to any specific wand device and could even be used with any combination
of input/controller/joystick device and an orientation tracker.
"""

import viz
import vizmat
import vizact

import transportation


class WandMagicCarpet(transportation.AccelerationTransport):
	"""Creates a new instance of the magic carpet.

	@param orientationTracker: a linkable node, only orientation information will be sampled.
	@param debug: if True a sphere will be created at the center of the magic carpet.
	@param parentedTracker: Should be set to True if the orientation values of the orientationTracker are affected by the transport's orientation
	"""
	def __init__(self, orientationTracker,
					node=None,
					usingPhysics=False,
					acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
					maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
					rotationAcceleration=90.0, # in degrees per second per second
					maxRotationSpeed=120.0, # in degrees per second
					autoBreakingDragCoef=0.1, # determines how quickly the transport will stop
					dragCoef=0.0001,
					rotationAutoBreakingDragCoef=0.2, # determines how quickly the transport will stop
					rotationDragCoef=0.0001, # normal drag coef
					parentedTracker=False,
					**kwargs):

		self._exp = 2.0
		self._orientationTracker = orientationTracker
		self._usingPhysics = usingPhysics

		self._physManualEuler = [0, 0, 0]
		self._physLink = None
		self._physNode = None
		self._postPhysicsUpdateEvent = None
		self._parentedTracker = parentedTracker

		# check if we're using physics and if so add a collision object
		if usingPhysics:
			# add a node if there is none provided
			if node is None:
				node = viz.addGroup()
			self._node = node
			self.__enablePhysics()

		# init the base class
		super(WandMagicCarpet, self).__init__(node=node,
												acceleration=acceleration,
												maxSpeed=maxSpeed,
												rotationAcceleration=rotationAcceleration,
												maxRotationSpeed=maxRotationSpeed,
												autoBreakingDragCoef=autoBreakingDragCoef,
												dragCoef=dragCoef,
												rotationAutoBreakingDragCoef=rotationAutoBreakingDragCoef,
												rotationDragCoef=rotationDragCoef,
												**kwargs)

		self.setUpdateFunction(lambda transport: None)

	def getTracker(self):
		"""Returns the tracker used to determine the orientation used as a
		reference frame for acceleration of the transport

		@return viz.VizNode()
		"""
		return self._orientationTracker

	def replaceTracker(self, tracker):
		"""Replaces the orientation tracker used by the wand wrapper.

		@param tracker: a linkable node, only orientation information will be sampled.
		"""
		self._orientationTracker = tracker

	def moveForward(self, mag=1):
		"""Moves forward"""
		self._Ap[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def moveBackward(self, mag=1):
		"""Moves backward"""
		self._Ap[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def moveLeft(self, mag=1):
		"""Moves left"""
		self._Ap[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def moveRight(self, mag=1):
		"""Moves right"""
		self._Ap[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def moveUp(self, mag=1):
		"""Moves up"""
		self._Ap[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def moveDown(self, mag=1):
		"""Moves down"""
		self._Ap[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def turnLeft(self, mag=1):
		"""Accelerate yaw rotation"""
		self._Ar[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def turnRight(self, mag=1):
		"""Accelerate yaw rotation"""
		self._Ar[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def lookDown(self, mag=1):
		"""Accelerates pitch rotation"""
		self._Ar[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def lookUp(self, mag=1):
		"""Accelerates pitch rotation"""
		self._Ar[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()

	def getAccelerationRotationMatrix(self):
		"""Returns the acceleration rotation matrix, i.e. the matrix used to
		determine reference frame for acceleration.

		@return vizmat.Transform()
		"""
		rotMat = vizmat.Transform()
		if self._orientationTracker is None:
			rotMat.setEuler([self.getEuler()[0], 0, 0])
		elif self._parentedTracker:
			parents = self.getParents()
			if parents:
				rotMat.setEuler([self._orientationTracker.getEuler()[0]-parents[0].getEuler()[0], 0, 0])
			else:
				rotMat.setEuler([self._orientationTracker.getEuler()[0], 0, 0])
		else:
			rotMat.setEuler([self._orientationTracker.getEuler()[0]+self.getEuler()[0], 0, 0])
		return rotMat

	def getPhysicsNode(self):
		"""This method returns the node used for physics. The transport node
		is linked to this object. Only applies if physics is enabled.

		@return viz.VizNode()
		"""
		return self._physNode

	def getPhysicsLink(self):
		"""This method returns the link between the physics node and the
		transport

		@return viz.VizLink()
		"""
		return self._physLink

	def finalize(self):
		"""Function which executes the quequed functions such as
		moveForward and moveBack basing them off the sample orientation
		from the tracker. Should be called regularly either by a timer
		or ideally at every frame.
		"""
		#print self._Ap
		if self._usingPhysics:
			transportation.Transport.finalize(self)
			self._updateTime()
			idt = min(60.0, 1.0/self._dt)

			# if necessary normalize the acceleration
			mag = self._Ap.length()
			if mag > 1.0:
				self._Ap = self._Ap / mag
			# .. and for rotation
			mag = self._Ar.length()
			if mag > 1.0:
				self._Ar = self._Ar / mag

			# scale acceleration (right now no units just 0-1 range magnitude vector)
			self._Ap *= self._acceleration
			# .. and for rotation
			self._Ar *= self._rotationAcceleration

			rotMat = self.getAccelerationRotationMatrix()

			invMat = rotMat.inverse()

			# we want to have a fast deceleration if we're not moving in a particular direction
			breakingVec = vizmat.Vector(invMat.preMultVec(self._Vp)) * self._autoBreakingDragCoef * idt
			localVp = invMat.preMultVec(self._Vp)
			for i in range(0, 3):
				if self._Ap[i] != 0 and (self._Ap[i]*localVp[i] > 0):
					breakingVec[i] = 0
				if breakingVec[i]:
					if self._autoBreakingTimeoutCounter[i] < self._autoBreakingTimeout:
						breakingVec[i] = 0# cancel breaking
					self._autoBreakingTimeoutCounter[i] += self._dt
				else:
					self._autoBreakingTimeoutCounter[i] = 0
			breakingVec = rotMat.preMultVec(breakingVec)

			# now apply the acceleration to the velocity
			drag = self._Vp * self._dragCoef * idt
			adjAp = rotMat.preMultVec(self._Ap)

			self._Vp[0] += (adjAp[0] - drag[0] - breakingVec[0]) * self._dt
			self._Vp[1] += (adjAp[1] - drag[1] - breakingVec[1]) * self._dt
			self._Vp[2] += (adjAp[2] - drag[2] - breakingVec[2]) * self._dt
			velMag = self._Vp.length()
			if velMag > self._maxSpeed:
				self._Vp = (self._Vp / velMag) * self._maxSpeed

			# .. and for rotation
			breakingVec = self._Vr * self._rotationAutoBreakingDragCoef * idt
			for i in range(0, 3):
				if self._Ar[i] != 0:
					breakingVec[i] = 0
				if breakingVec[i]:
					if self._rotationAutoBreakingTimeoutCounter[i] < self._rotationAutoBreakingTimeout:
						breakingVec[i] = 0# cancel breaking
					self._rotationAutoBreakingTimeoutCounter[i] += self._dt
				else:
					self._rotationAutoBreakingTimeoutCounter[i] = 0

			drag = self._Vr * self._rotationDragCoef * idt
			self._Vr[0] += (self._Ar[0] - drag[0] - breakingVec[0]) * self._dt
			self._Vr[1] += (self._Ar[1] - drag[1] - breakingVec[1]) * self._dt
			velMag = self._Vr.length()
			if velMag > self._maxRotationSpeed:
				self._Vr = (self._Vr / velMag) * self._maxRotationSpeed

			# .. and for rotation
			self._physManualEuler[0] += self._Vr[0] * self._dt
			self._physManualEuler[1] += self._Vr[1] * self._dt
			self._physManualEuler[2] += self._Vr[2] * self._dt

			self._physManualEuler[1] = max(-90, min(90, self._physManualEuler[1]))

#			directedSpeedScale = rotMat.preMultVec(self._movementSpeed)
#			scaledSpeed = [self._Vp[0]*directedSpeedScale[0], self._Vp[1]*directedSpeedScale[1], self._Vp[2]*directedSpeedScale[2]]
			self._physNode.setVelocity(self._Vp)

			self._Ap[0] = 0
			self._Ap[1] = 0
			self._Ap[2] = 0
			self._Ar[0] = 0
			self._Ar[1] = 0
			self._Ar[2] = 0
		else:
			super(WandMagicCarpet, self).finalize()

	def _postPhysics(self):
		"""Internal method that is run post physics in order to stabilize euler angles, etc."""
		self._physNode.setEuler(self._physManualEuler)
		self._Vp = vizmat.Vector(self._physNode.getVelocity())

		self._physNode.setAngularVelocity([0, 0, 0])
		self._node.getMatrix()

	def remove(self):
		"""Removes the transport, if using physics, will also clean up any
		added physics objects.
		"""
		super(WandMagicCarpet, self).remove()
		self.__disablePhysics()

	def __disablePhysics(self):
		"""Internal method that disables physics on transports"""
		if self._postPhysicsUpdateEvent is not None:
			self._postPhysicsUpdateEvent.remove()
			self._postPhysicsUpdateEvent = None
			self._physNode.remove()
			self._physNode = None
			self._physLink.remove()
			self._physLink = None

	def __enablePhysics(self):
		"""Internal method that enables physics on transports"""
		# add a node for physics
		physicsHeightOffset = 0.1
		self._physNode = viz.addGroup()
		pos = self._node.getPosition()
		pos[1] += physicsHeightOffset*2.0
		self._physNode.setPosition(pos)
		collideBox = self._physNode.collideBox([1.0, physicsHeightOffset, 1.0])
		collideBox.setBounce(0.0001)
		collideBox.setFriction(0.00)

		# add a link to the transport
		self._physLink = viz.link(self._physNode, self._node)
		self._physLink.preTrans([0, -physicsHeightOffset/2.0, 0])

		self._postPhysicsUpdateEvent = vizact.onupdate(viz.PRIORITY_PHYSICS+1, self._postPhysics)


if __name__ == "__main__":
	viz.go()

	# load a model
	piazza = viz.add('piazza.osgb')

	transportRepresentation = viz.add("beachball.osgb")
	wand = WandMagicCarpet(orientationTracker=None,
					node=None,
					usingPhysics=False,
					acceleration=5.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
					maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
					rotationAcceleration=90.0, # in degrees per second per second
					maxRotationSpeed=120.0, # in degrees per second
					autoBreakingDragCoef=0.1, # determines how quickly the transport will stop
					dragCoef=0.0001,
					rotationAutoBreakingDragCoef=0.2, # determines how quickly the transport will stop
					rotationDragCoef=0.0001, # normal drag coef
					parentedTracker=False,
					autoBreakingTimeout=0.2, # how long before auto breaking is enabled * see comment
					rotationAutoBreakingTimeout=0.2, # how long before rotational auto breaking is enabled * see comment
					)
	# Note regarding autoBreakingTimeouts:
	# When using a transport, the auto breaking function acts as if the user has
	# applied pressure to a break peddle. Auto breaking occurs whenever the user
	# stops applying acceleration in a particular direction. So if you remove your
	# finger from the forward key, the auto breaking coefficient in the forward
	# direction is automatically applied. If you'd like to coast to a stop, you
	# can lower or remove (set to 0) the auto breaking coefficient.
	#
	# When using an update function with a transport, the state of they buttons/
	# keys are constantly sampled. If the data is irregular, it may be best to
	# set an autoBreakingTimeout, which is a minimum time to wait for input signal
	# before auto breaking is used.
	#
	# If you're not setting an update function, but manually calling the moveForward,
	# moveBackward, etc events, then it's generally a good idea to apply some timeout
	# as events triggered by keyboards and other inputs tend to update less than
	# once per frame.

	vizact.onkeydown(viz.KEY_UP, wand.moveForward)
	vizact.onkeydown(viz.KEY_DOWN, wand.moveBackward)
	vizact.onkeydown(viz.KEY_LEFT, wand.moveLeft)
	vizact.onkeydown(viz.KEY_RIGHT, wand.moveRight)

	vizact.onkeydown('w', wand.lookUp)
	vizact.onkeydown('s', wand.lookDown)
	vizact.onkeydown('a', wand.turnLeft)
	vizact.onkeydown('d', wand.turnRight)
	vizact.onkeydown('z', wand.moveDown)
	vizact.onkeydown('x', wand.moveUp)

	base = viz.addGroup()
	box = viz.add("arrow.wrl")
	box.setPosition(0, 0, 1.5)
	box.setScale([0.15]*3)
	box.setParent(base)

	vizact.onkeydown('5', wand.setNode, base)

	group = viz.addGroup()
	group.setParent(wand)
	group.setEuler([0, 0, 0])
	link = viz.link(group, viz.MainView)
	link.setSrcFlag(viz.ABS_GLOBAL)
#	link = viz.link(wand, viz.MainView)
